{

  ScrobblerUtils, Version 0.1b

  A Borland/Codegear/Embarcadero Delphi-Class for Last.Fm scrobbling support.
  See http://www.last.fm in case you dont know what "scrobbling" means

}
{

  Read and accept the "Terms of Use" on LastFM before using this unit.

  http://www.last.fm/api/tos

  -------------------------------------------------------

  The contents of this file are subject to the Mozilla Public License
  Version 1.1 (the "License"); you may not use this file except in
  compliance with the License. You may obtain a copy of the License at
  http://www.mozilla.org/MPL/

  Software distributed under the License is distributed on an "AS IS"
  basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
  License for the specific language governing rights and limitations
  under the License.

  The Original Code is ScrobblerUtils.

  The Initial Developer of the Original Code is Daniel Gaussmann,
  mail@gausi.de. Portions created by the Initial Developer are
  Copyright (C) 2008 the Initial Developer. All Rights Reserved.

  Contributor(s): (none yet)

  Alternatively, the contents of this file may be used under the terms
  of the GNU Lesser General Public License Version 2.1 or later
  (the  "LGPL"), in which case the provisions of LGPL are applicable
  instead of those above. If you wish to allow use of your version of
  this file only under the terms of the LGPL and not to allow others to use
  your version of this file under the MPL, indicate your decision by
  deleting the provisions above and replace them with the notice and
  other provisions required by the LGPL. If you do not delete
  the provisions above, a recipient may use your version of this file
  under either the MPL or the LGPL License.

  -------------------------------------------------------
}

{
  Version History:
  -------------------------------------------------------

  v0.1b, november 2009:
  ---------------------
    * fixed some PChar/PAnsiChar-stuff with the log-messages.

  v0.1a, march 2009:
  ------------------
    * replaced some "PChar" by "PAnsiChar"
    
  v0.1, march 2009:
  -----------------
    * First release

  -------------------------------------------------------
}

unit ScrobblerUtils;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes,
  IdBaseComponent, IdComponent, IdTCPConnection, IdTCPClient, IdHTTP, IdStack, IdException,
  Dialogs, md5, StdCtrls, ShellApi, DateUtils, IniFiles, Contnrs;

const
    UnixStartDate: TDateTime = 25569.0;

    BaseApiURL = 'http://ws.audioscrobbler.com/2.0/';
    BaseScrobbleURL = 'http://post.audioscrobbler.com/';

    Scrobble_ProtocolError = 'Protokollfehler.';
    Scrobble_ConnectError = 'Verbindungsfehler. Bitte berprfen Sie ihre Interneteinstellungen.';
    Scrobble_UnkownError = 'Ein unbekannter Fehler ist aufgetreten.';
    ScrobbleFailureWait = 'Da stimmt was nicht mit dem Scrobbeln. Log-Eintrge lesen fr weitere Details.';

    WM_Scrobbler = WM_User + 850;

    SC_HandShakeError = 1;
    SC_HandShakeCancelled = 2;
    SC_HandShakeException = 3;
    SC_NowPlayingError = 4;
    SC_NowPlayingException = 5;
    SC_SubmissionError = 6;
    SC_SubmissionException = 7;
    SC_GetToken = 10;
    SC_GetTokenException = 11;
    SC_GetSession = 12;
    SC_GetSessionException = 13;
    SC_HandShakeComplete = 14;
    SC_NowPlayingComplete = 15;
    SC_SubmissionComplete = 16;
    SC_SubmissionSkipped = 17;
    SC_BeginWork = 50;
    SC_EndWork = 51;
    SC_TooMuchErrors = 100;
    SC_Message = 1000;
    
type

    TScrobbleStatus = (hs_OK=1, hs_EXCEPTION, hs_UnknownFailure, hs_UNAUTHORIZED, hs_BANNED, hs_BADAUTH, hs_BADTIME, hs_FAILED, hs_BADSESSION, hs_HANDSHAKECANCELLED);

const

    ScrobbleStatusStrings: Array[1..10] of AnsiString =
    ('(OK)',
    '(Exception)',
    '(Unknown failure.)',
    'Unauthorized. Bitte Scrobbeln erst konfigurieren. ',
    'Client gebannt. Bitte besorgen Sie sich ein Update des Programms.',
    'Bad Authorization. Bitte Scrobbeln neu konfigurieren. ',
    'Bad Time. Bitte stellen Sie die Uhrzeit auf ihrem System richtig ein.',
    'Server Failure. Bitte versuchen Sie es spter erneut.',
    'Bad Session. Neuer Handshake bentigt.',
    'Zu viele Fehler. Scrobbeln wird fr eine gewisse Zeit deaktiviert.'
    );

type

    TScrobbleFile = Class
          Interpret: UTF8String;
          Title: UTF8String;
          Album: UTF8String;
          Length: UTF8String;    // Lnge des Tracks in Sekunden, oder '' falls unbekannt
          TrackNr: UTF8String;   // TrackNumber, '' falls unbekannt
          MBTrackID: UTF8String; // Music-Brainz-TrackID. Hier: immer '' ;-)

          DisplayTitle: UTF8String;
          // zustzlich frs Submitten:
          StartTime: UTF8String;
          Source: UTF8String;
          Rating: UTF8String;

          procedure Assign(aScrobbleFile: TScrobbleFile);
    end;

    // Klasse TScrobbler.
    // Wird vom ApplicationScrobbler fr jeweils einen Job erzeugt.
    // TScrobbler startet dann einen eigenen Thread, der den gewnschten Job
    // ausfhrt und diverse Messages an die Form sendet
    // Anschlieend gibt sich der Scrobbler selbst frei.
    TScrobbler = class
      private
          // im Constructor setzen
          p: AnsiString; //  = 'p=1.2.1' // Protokoll-Version
          c: AnsiString; // Client-ID
          v: AnsiString; // Client-Version

          fWindowHandle: HWND; // Handle zum Scrobble-Fenster zum Schicken der Nachrichten (Parameter im Create)
          fThread: Integer;

          fIDHttp: TIdHttp;
          // --------------------------------------
          ApiKey: AnsiString;
          Secret: AnsiString;
          Token: AnsiString;   // ber GetToken ermitteln lassen

          // GetSession
          Username: AnsiString;   // Name des LastFM-Users
          SessionKey: AnsiString; // Session-Key frs Scrobblen. Muss (einmalig) ber diverse Funktionen angefordert werden.

          // frs "NowPlaying"
          SessionID: AnsiString;
          NowPlayingURL: AnsiString;
          SubmissionURL: AnsiString;

          ParamList: TStrings;

          SuccessMessage: UTF8String;
          EarliestNextHandshake: TDateTime;
          // -----------------------------------------

          //Die "f"-Methoden werden in einem separaten Thread ausgefhrt.

          // Authentifizierung
          // -----------------------------------
          // GetToken: Schritt 1
          // Liefert ein Token zurck. Dieses ist ca. eine Stunde gltig
          // und muss vom LastFM-User authentifiziert werden.
          // Dies Geschieht im Browser ber einen Speziellen Link, der
          // eben dieses Token enthlt
          // Darber wird der Anwendung erlaubt, im Userprofil zu arbeiten.
          procedure fGetToken;

          // Schritt 2: Nutzer klickt im Browser auf "OK - Anwendung darf"

          // GetSession: Schritt 3
          // Das vorher authentifizierte Token wird benutzt, um einen
          // Session-KEY zu erhalten. In der Antwort vom LastFM-Server
          // ist der Username und eben dieser Session-Key enthalten.
          // Der Session-Key ist beliebig lange gltig. und kann z.B.
          // in einer Ini-Datei gespeichert werden und spter wiederverwendet
          // werden
          // Der User kann die Anwendung in seinem Profil deaktivieren - dann
          // erst wird der Session-Key ungltig
          procedure fGetSession;


          // Scrobbling
          // ---------------------------------------
          // Handshake:
          // Username und Session-Key werden benutzt, um eine Scrobbling-Session
          // zu starten.
          // Antwort enthlt eine Session-ID, sowie zwei Links zur weiteren
          // Kommunikation. Ist einmal beim Start der Anwendung ntig
          // Fehler:
          // BANNED: Die Anwendung (d.h. der Programmierer ;-)) hat zu oft Mist gebaut und darf nicht mehr mit LastFM spielen. => Update der Anwendung ntig
          // BADAUTH: Wahrscheinlich Username, Session-KEY, oder auth-Token (gebildet aus "Secret")
          //          falsch. => Neu authentifizieren
          // BADTIME: Uhr geht falsch. => User soll Uhr neu stellen.
          // FAILED <reason>: Server-Fehler => Spter nochmal probieren
          function fPerformHandShake: TScrobbleStatus;
          function ParseHandShakeResult(aResponse: Ansistring): TScrobbleStatus;

          // NowPlaying:
          // Session-ID und eine der Links wird verwendet, um Daten ber das
          // aktuelle AudioFile zu senden
          // Fehler:
          // BADSESSION: Session-ID ungltig. => Neuer Handshake.
          function fPerformNowPlayingNotification: TScrobbleStatus;
          function fPerformSubmissionNotification: TScrobbleStatus;

          function ParseNowPlayingResult(aResponse: Ansistring): TScrobbleStatus;
          function ParseSubmissionResult(aResponse: Ansistring): TScrobbleStatus;

          procedure fScrobbleNowPlaying;
          procedure fScrobbleSubmit;
          //----------------------------------

          // Startet Thread
          procedure GetToken;
          procedure GetSession;
          procedure ScrobbleNowPlaying;
          procedure ScrobbleSubmit;

          function GetUTCTimeStamp: AnsiString;
      public
          constructor Create(aHandle: HWnd);
          destructor Destroy; override;
          // Nichts weiter
          // TScrobbler wird nur von TApplicationScrobbler erstellt und benutzt.
    end;


    // Klasse TApplicationScrobbler
    // Diese Klasse ist so ausgelegt, dass sie _relativ_ einfach in einen
    // bestehenden Player integriert werden kann.
    // Folgendes ist zu tun:
    // 1.) Im Player-Fenster die Scrobble-Messages abfangen und verarbeiten
    //     Dazu kann man fast komplett die Methode aus der Demo verwenden
    // 2.) Ein Setup-System bauen, um den Anwender anzuleiten die drei 
    //     Schritte zur Authentifizierung durchzufhren
    // 3.) Wenn der PLayer stoppt, pausiert, anfngt zu spielen oder das Lied
    //     wechselt, mssen die entsprechenden ApplicationScrobbler-Methoden
    //     aufgerufen werden
    //     Liedwechsel............ChangeCurrentPlayingFile
    //     Wiedergabestart........PlaybackStarted
    //     Pause..................PlaybackPaused
    //     Wiedergabe fortsetzen..PlaybackResumed
    //     Wiedergabe stoppen.....AddToScrobbleList(). Der Parameter gibt an, ob
    //                              der Player VOR dem Stoppen noch gespielt hat
    //                             (wichtig fr die Berechnung der Wiedergabedauer).
    //     Nchster Titel.........Stop - NeuerTitel - Play. ;-)
    // 4.) Fertig. :D
    TApplicationScrobbler = class
        private
            fMainWindowHandle: HWND;
            ErrorInARowCount: Integer;

            StartTimeUTC: TSystemTime;
            StartTimeLocal: TDateTime;
            PlayAmount: Integer; // in Sekunden

            // Gibt an, ob gerade gescrobbelt wird.
            fWorking: Boolean;
            FScrobbleJobCount: Integer;

            fNewFile: Boolean; // gibt an, ob was neues da ist: Knnte ja sein, dass beim NP-Scrobbeln das File direkt neu gesetzt wird.
            fPlayingFile: TScrobbleFile;
            fCurrentFileAdded: Boolean;

            EarliestNextHandshake: TDateTime;
            TimeToWaitInterval: Integer;

            // Setzt fWorking um und setzt eine Message ans Fenster ab.
            procedure BeginWork;
            procedure EndWork;

            procedure ScrobbleCurrentFile;
            procedure ScrobbleJobList;


        public
            ClientID: AnsiString;
            ClientVersion: AnsiString;
            SessionKey: AnsiString; // Der quasi allzeit gltige Session-Key    // "Sicher" in Ini-Datei speichern
            Username: AnsiString;   // Der Username des LastFM-Users            //

            AlwaysScrobble: Boolean; // Immer Scrobbeln, d.h. direkt beim Start der Anwendung aktivieren
            DoScrobble: Boolean;     // Laufzeit-Variable

            Token: AnsiString;            // getToken-Ergebnis. Muss der LastFM-User authentifizieren
            SessionID: AnsiString;        // Erhlt man beim Handshake
            NowPlayingURL: AnsiString;    // Erhlt man beim Handshake
            SubmissionURL: AnsiString;    // Erhlt man beim Handshake

            JobList: TObjectList;         // Liste, in der die zu scrobbelnden Lieder gepuffert werden
            LogList: TStrings;

            ApiKey: AnsiString;           // ApiKey und Secret.
            Secret: AnsiString;           // bekommt man von LastFM

            property Working: Boolean Read fWorking; 
            constructor Create(aHandle: HWND);
            destructor Destroy; override;

            procedure LoadFromIni(Ini: TMemIniFile);
            procedure SaveToIni(Ini: TMemIniFile);

            // Beim Ende eines Authorisierungs-Jobs im Message-Handler ausfhren!
            procedure JobDone;

            procedure GetToken;
            procedure GetSession;

            // Wenn man was gescrobbelt hat, knnte in der Zwischenzeit was neues dazu gekommen sein, oder sich gendert haben
            procedure ScrobbleNext(PlayerIsPlaying: Boolean);
            procedure ScrobbleNextCurrentFile(PlayerIsPlaying: Boolean);

            // wird in den Message-Handlern bei Fehlern aufgerufen
            Procedure ScrobbleAgain(PlayerIsPlaying: Boolean);

            function AddToScrobbleList(IsJustPlaying: Boolean): Boolean; // result: hinzugefgt oder nicht

            procedure ChangeCurrentPlayingFile(Interpret, Title, Album: WideString; Length, TrackNr: Integer);

            procedure HandleHandshakeFailure(aFailure: TScrobbleStatus);
            procedure CountError(aFailure: TScrobbleStatus);
            procedure CountSuccess;

            procedure PlaybackStarted;
            procedure PlaybackPaused;
            procedure PlaybackResumed;

            procedure AllowHandShakingAgain;
    end;




  procedure StartGetToken(Scrobbler: TScrobbler);
  procedure StartGetSession(Scrobbler: TScrobbler);

  procedure StartScrobbleNowPlaying(Scrobbler: TScrobbler);
  procedure StartScrobbleSubmit(Scrobbler: TScrobbler);

  //Ermittelt aus einer Server-Antwort das Token
  function GetTokenFromResponse(aResponseString: AnsiString): AnsiString;
  // Ermittelt aus einer Server-Antwort den Usernamen des LastFM-Accounts
  function GetUserNameFromResponse(aResponseString: AnsiString): AnsiString;
  // Ermittelt den (allzeit gltigen) Session-Key aus der Server-Antwort
  function GetSessionKeyFromResponse(aResponseString: AnsiString): AnsiString;


implementation


// Einige lose Funktionen. Werden im Hauptthread ausgefhrt und liefern der Anwendung
// Daten fr den weiteren Verlauf des "Scrobbelns"

// Ermitteln eines Tokens. Erwartetes Format:
            {
             <?xml
            version="1.0"
            encoding="utf-8"?>
            <lfm
            status="ok">
            <token>  ...some md5-hash... </token></lfm>
            }
function GetTokenFromResponse(aResponseString: AnsiString): AnsiString;
var idx: Integer;
begin
    idx := Pos('<token>', aResponseString );
    if idx >= 1 then
        result := copy(aResponseString, idx + length('<token>'), 32)
    else
        result := '';
end;

// Ermittelt aus einer Server-Antwort den Usernamen des LastFM-Accounts. Format:
            {
            <?xml
            version="1.0"
            encoding="utf-8"?>
            <lfm
            status="ok">
            <session>
            <name> ...username... </name>
            <key> ...some md5-hash... </key>   
            <subscriber>0</subscriber>
            </session></lfm>
            }
function GetUserNameFromResponse(aResponseString: AnsiString): AnsiString;
var start, ende: Integer;
begin
    start := Pos('<name>', aResponseString);
    ende  := Pos('</name>', aResponseString);
    if (start < ende) and (ende > 1) then
        result := copy(aResponseString, start + length('<name>'), ende - (start + length('<name>')) )
    else
        result := '';
end;
  // Ermittelt den (allzeit gltigen) Session-Key aus der Server-Antwort. Format wie bei Username
function GetSessionKeyFromResponse(aResponseString: AnsiString): AnsiString;
var idx: Integer;
begin
    idx := Pos('<key>', aResponseString );
    if idx >= 1 then
        result := copy(aResponseString, idx + length('<key>'), 32)
    else
        result := '';
end;


// von den Indys abgeguckt, und um '&' erweitert
function ParamsEncode(const ASrc: Ansistring): AnsiString;
var i: Integer;
begin
  Result := '';
  for i := 1 to Length(ASrc) do
  begin
    if (ASrc[i] in ['&', '*','#','%','<','>',' ','[',']'])
       or (not (ASrc[i] in [#33..#128]))
    then
    begin
      Result := Result + '%' + IntToHex(Ord(ASrc[i]), 2);
    end
    else
    begin
      Result := Result + ASrc[i];
    end;
  end;
end;


procedure TScrobbleFile.Assign(aScrobbleFile: TScrobbleFile);
begin
    if assigned(aScrobbleFile) then
    begin
        Interpret    := aScrobbleFile.Interpret    ;
        Title        := aScrobbleFile.Title        ;
        Album        := aScrobbleFile.Album        ;
        Length       := aScrobbleFile.Length       ;
        TrackNr      := aScrobbleFile.TrackNr      ;
        MBTrackID    := aScrobbleFile.MBTrackID    ;
        DisplayTitle := aScrobbleFile.DisplayTitle ;

        StartTime    := aScrobbleFile.StartTime    ;
        Source       := aScrobbleFile.Source       ;
        Rating       := aScrobbleFile.Rating       ;
    end;
end;


// -------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------
// Klasse TApplicationScrobbler
// -------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------


constructor TApplicationScrobbler.Create(aHandle: HWND);
begin
    inherited Create;
    fMainWindowHandle := aHandle;
    LogList := TStringList.Create;
    JobList := TObjectList.Create(True);
    fPlayingFile := TScrobbleFile.Create;
    fCurrentFileAdded := False;
    ErrorInARowCount := 0;
    EarliestNextHandshake := Now;
    TimeToWaitInterval := 1;

    DoScrobble := False;
end;
destructor TApplicationScrobbler.Destroy;
begin
    LogList.Free;
    JobList.Free;
    fPlayingFile.Free;
    inherited destroy;
end;

procedure TApplicationScrobbler.BeginWork;
begin
    fWorking := True;
    SendMessage(fMainWindowHandle, WM_Scrobbler, SC_BeginWork, 0);
end;

procedure TApplicationScrobbler.EndWork;
begin
    fWorking := False;
    SendMessage(fMainWindowHandle, WM_Scrobbler, SC_EndWork, 0);
end;

procedure TApplicationScrobbler.JobDone;
begin
    EndWork;
end;

procedure TApplicationScrobbler.LoadFromIni(Ini: TMemIniFile);
begin
    // Der quasi allzeit gltige Session-Key
    SessionKey := Ini.ReadString('Scrobbler', 'SessionKey', '');
    Username   := Ini.ReadString('Scrobbler', 'Username', '');
    if (Username = '') or (SessionKey = '') then
    begin
        LogList.Add('Loading Settings: No username/sessionkey found.');
    end
    else
    begin
        LogList.Add('Username: ' + Username);
        LogList.Add('Sessionkey: ' + Sessionkey);
    end;

    AlwaysScrobble := Ini.ReadBool('Scrobbler', 'AlwaysScrobble', False);
    DoScrobble := AlwaysScrobble;
end;
procedure TApplicationScrobbler.SaveToIni(Ini: TMemIniFile);
begin
    Ini.WriteString('Scrobbler', 'SessionKey', SessionKey);
    Ini.WriteString('Scrobbler', 'Username', Username);
    Ini.WriteBool('Scrobbler', 'AlwaysScrobble', AlwaysScrobble);
end;

procedure TApplicationScrobbler.CountError(aFailure: TScrobbleStatus);
begin
    inc(ErrorInARowCount);
    if ErrorInARowCount >= 10 then // 3
    begin
        // Zurckfallen zum Handshake: ID auf '' setzen.
        SessionID := '';
        SendMessage(fMainWindowHandle, WM_Scrobbler, SC_TooMuchErrors, 0);
    end;
end;

procedure TApplicationScrobbler.CountSuccess;
begin
    ErrorInARowCount := 0;
    TimeToWaitInterval := 1;
end;

procedure TApplicationScrobbler.HandleHandshakeFailure(aFailure: TScrobbleStatus);
begin
    SessionID := '';
    if aFailure <> hs_Ok then
    begin
        EarliestNextHandshake := IncMinute(Now, TimeToWaitInterval);
        SendMessage(fMainWindowHandle, WM_Scrobbler, SC_Message, LParam(PChar('Scrobbling fr ca. ' + IntToStr(TimeToWaitInterval) + ' Minuten deaktiviert.'  )));

        if TimeToWaitInterval < 60 then
            TimeToWaitInterval := TimeToWaitInterval * 2
        else
            TimeToWaitInterval := 120;
    end;
    EndWork;
end;

function TApplicationScrobbler.AddToScrobbleList(IsJustPlaying: Boolean): Boolean; // result: hinzugefgt oder nicht
var aScrobbleFile: TScrobbleFile;
    StartTimeDelphi: TDateTime;
    diff: LongInt;
begin
    result := False;
    if IsJustPlaying then
        PlayAmount := PlayAmount + SecondsBetween(Now, StartTimeLocal);

    if DoScrobble then
    begin
        if ((PlayAmount > 240) or (PlayAmount > ( StrToIntDef(fPlayingFile.Length, 0) Div 2 )) )
           AND
           (StrToIntDef(fPlayingFile.Length, 0) > 30)
        then
        begin
            if (not fCurrentFileAdded) then   // hier beim else nichts senden, daher so. ;-)
            begin
                aScrobbleFile := TScrobbleFile.Create;
                aScrobbleFile.Assign(fPlayingFile);
                StartTimeDelphi := EncodeDateTime(StartTimeUTC.wYear, StartTimeUTC.wMonth, StartTimeUTC.wDay, StartTimeUTC.wHour, StartTimeUTC.wMinute, StartTimeUTC.wSecond, StartTimeUTC.wMilliSeconds);
                diff := Round((StartTimeDelphi - UnixStartDate) * 86400); // 86400: Sekunden pro Tag
                aScrobbleFile.StartTime := IntToStr(Diff);

                JobList.Add(aScrobbleFile);
                fCurrentFileAdded := True;
                Result := True;
                if not fWorking then
                begin
                    // Fang mit dem scrobbeln der Jobliste an!
                    ScrobbleJobList;
                end;
            end;
        end else
        begin
            if (not fCurrentFileAdded) then
            begin
                fCurrentFileAdded := True;  // so tun als ob. Sonst kommt diee Message evt. mehrfach
                // Scrobbeln verweigert, weil zu kurz...
                if (StrToIntDef(fPlayingFile.Length, 0) <= 30) then
                    SendMessage(fMainWindowHandle, WM_Scrobbler, SC_SubmissionSkipped, Lparam(PAnsiChar('Submission skipped: Datei zu kurz (min. 30 Sekunden)')))
                else
                    SendMessage(fMainWindowHandle, WM_Scrobbler, SC_SubmissionSkipped, Lparam(PAnsiChar('Submission skipped: Wiedergabe zu kurz  (min. 50% oder 4 Minuten).')));
            end;
        end;
    end;
end;

procedure TApplicationScrobbler.ChangeCurrentPlayingFile(Interpret, Title, Album: WideString; Length, TrackNr: Integer);
var s: TSystemTime;
    StartTimeDelphi: TDateTime;
    diff: LongInt;
begin
    GetSystemTime(s);
    fPlayingFile.DisplayTitle := Utf8Encode(Interpret + ' - ' + Title);

    fPlayingFile.Interpret := ParamsEncode(Utf8Encode(Interpret));
    fPlayingFile.Title     := ParamsEncode(Utf8Encode(Title));
    fPlayingFile.Album     := ParamsEncode(Utf8Encode(Album));
    fPlayingFile.Length    := IntToStr(Length);

    if TrackNr <> 0 then
        fPlayingFile.TrackNr := IntToStr(TrackNr)
    else
        fPlayingFile.TrackNr := '';

    StartTimeDelphi := EncodeDateTime(s.wYear, s.wMonth, s.wDay, s.wHour, s.wMinute, s.wSecond, s.wMilliSeconds);
    diff := Round((StartTimeDelphi - UnixStartDate) * 86400); // 86400: Sekunden pro Tag
    fPlayingFile.StartTime := IntToStr(diff);

    fPlayingFile.Rating := '';  // oder 'L' fr "Love"?
    fPlayingFile.Source := 'P'; // Bei Webstreams: 'R'

    fNewFile := True;
    fCurrentFileAdded := False;
end;


procedure TApplicationScrobbler.GetToken;
var aScrobbler: TScrobbler;
begin
    BeginWork;
    aScrobbler := TScrobbler.Create(fMainWindowHandle);
    try
        aScrobbler.Secret := Secret;
        aScrobbler.ApiKey := ApiKey;
        aScrobbler.c := ClientID;
        aScrobbler.v := ClientVersion;
        aScrobbler.GetToken;
    except
        aScrobbler.Free;
        EndWork;
    end;
end;

procedure TApplicationScrobbler.GetSession;
var aScrobbler: TScrobbler;
begin
    BeginWork;
    aScrobbler := TScrobbler.Create(fMainWindowHandle);
    try
        aScrobbler.Secret := Secret;
        aScrobbler.ApiKey := ApiKey;
        aScrobbler.Token := Token;
        aScrobbler.c := ClientID;
        aScrobbler.v := ClientVersion;
        aScrobbler.GetSession;
    except
        aScrobbler.Free;
        EndWork;
    end;
end;

procedure TApplicationScrobbler.ScrobbleCurrentFile;
var aScrobbler: TScrobbler;
begin
    fNewFile := False;
    if DoScrobble then
    begin
        BeginWork;
        aScrobbler := TScrobbler.Create(fMainWindowHandle);
        try
            aScrobbler.Secret := Secret;
            aScrobbler.ApiKey := ApiKey;
            aScrobbler.EarliestNextHandshake := EarliestNextHandshake;
            aScrobbler.Username   := self.Username;
            aScrobbler.SessionKey := self.SessionKey;
            aScrobbler.SessionID  := SessionID;
            aScrobbler.c := ClientID;
            aScrobbler.v := ClientVersion;
            aScrobbler.ParamList.Add('s=' + SessionID);
            aScrobbler.ParamList.Add('a=' + fPlayingFile.Interpret);
            aScrobbler.ParamList.Add('t=' + fPlayingFile.Title    );
            aScrobbler.ParamList.Add('b=' + fPlayingFile.Album    );
            aScrobbler.ParamList.Add('l=' + fPlayingFile.Length   );
            aScrobbler.ParamList.Add('n=' + fPlayingFile.TrackNr  );
            aScrobbler.ParamList.Add('m=' + fPlayingFile.MBTrackID);
            aScrobbler.NowPlayingURL := NowPlayingURL;
            aScrobbler.SubmissionURL := SubmissionURL;
            aScrobbler.SuccessMessage := 'Now Playing Notification: OK. ' + fPlayingFile.DisplayTitle;
            aScrobbler.ScrobbleNowPlaying;
        except
            aScrobbler.Free;
            EndWork;
        end;
    end;
end;

procedure TApplicationScrobbler.ScrobbleNextCurrentFile(PlayerIsPlaying: Boolean);
begin
    if DoScrobble then
    begin
        if fNewFile and PlayerIsPlaying then
            ScrobbleCurrentFile
        else
            EndWork;
    end
    else
        EndWork;
end;


procedure TApplicationScrobbler.ScrobbleJobList;
var i: Integer;
    aScrobbleFile: TScrobbleFile;
    aScrobbler: TScrobbler;
begin
    // scrobbelcount setzen! Wichtig frs lschen aus der Joblist bei Erfolg
    FScrobbleJobCount := JobList.Count;
    if FScrobbleJobCount > 10 then
        FScrobbleJobCount := 10;
    // eigentlich sind bis zu 50 erlaubt
    // 10 am Stck reichen aber auch.
    // Der Rest wird dann ggf. danach gescrobbelt
    
    aScrobbler := TScrobbler.Create(fMainWindowHandle);
    try
          BeginWork;
          aScrobbler.Secret := Secret;
          aScrobbler.ApiKey := ApiKey;
          aScrobbler.Username    := Username;
          aScrobbler.SessionKey  := SessionKey;
          aScrobbler.SessionID   := SessionID;
          aScrobbler.c := ClientID;
          aScrobbler.v := ClientVersion;
          aScrobbler.EarliestNextHandshake := EarliestNextHandshake;

          aScrobbler.ParamList.Add('s=' + SessionID);
          for i := 0 to JobList.Count -1 do
          begin
              aScrobbleFile := TScrobbleFile(Joblist[i]);
              aScrobbler.ParamList.Add('a[' + IntToStr(i) + ']=' + aScrobbleFile.Interpret);
              aScrobbler.ParamList.Add('t[' + IntToStr(i) + ']=' + aScrobbleFile.Title    );
              aScrobbler.ParamList.Add('i[' + IntToStr(i) + ']=' + aScrobbleFile.StartTime);
              aScrobbler.ParamList.Add('o[' + IntToStr(i) + ']=' + aScrobbleFile.Source   );
              aScrobbler.ParamList.Add('r[' + IntToStr(i) + ']=' + aScrobbleFile.Rating   );
              aScrobbler.ParamList.Add('b[' + IntToStr(i) + ']=' + aScrobbleFile.Album    );
              aScrobbler.ParamList.Add('l[' + IntToStr(i) + ']=' + aScrobbleFile.Length   );
              aScrobbler.ParamList.Add('n[' + IntToStr(i) + ']=' + aScrobbleFile.TrackNr  );
              aScrobbler.ParamList.Add('m[' + IntToStr(i) + ']=' + aScrobbleFile.MBTrackID);
          end;
          aScrobbler.NowPlayingURL := NowPlayingURL;
          aScrobbler.SubmissionURL := SubmissionURL;

          if JobList.Count = 1 then
              aScrobbler.SuccessMessage := 'Scrobble: OK. ' + TScrobbleFile(Joblist[0]).DisplayTitle
          else
              aScrobbler.SuccessMessage := 'Scrobble: OK. (' + IntToStr(FScrobbleJobCount) + ' Dateien)';

          aScrobbler.ScrobbleSubmit;
    except
      aScrobbler.Free;
      EndWork;
    end;
end;

Procedure TApplicationScrobbler.ScrobbleNext(PlayerIsPlaying: Boolean);
var i: Integer;
begin
    // Diese Proc wird aufgerufen, wenn das Submitten per "ScrobbleJobList" erfolgreich war.
    // d.h.: Die gescrobbelten Dateien aus der JobListe entfernen
    for i := 1 to FScrobbleJobCount do
        JobList.Remove(JobList[0]);
    FScrobbleJobCount := 0;

    if DoScrobble then
    begin
        if JobList.Count > 0 then
            // erstmal die Liste weiter scrobblen
            ScrobbleJobList
        else
        begin
            if PlayerIsPlaying then
                ScrobbleCurrentFile  // muss nochmal gesendet werden, wird vom Submitten wohl berdeckt.
            else
                EndWork;
        end;
    end
    else
        EndWork;
end;

Procedure TApplicationScrobbler.ScrobbleAgain(PlayerIsPlaying: Boolean);
begin
    if DoScrobble then
    begin
        if JobList.Count > 0 then
            ScrobbleJobList
        else
        begin
            if PlayerIsPlaying then
                ScrobbleCurrentFile
            else
                EndWork;
        end;
    end
    else
        EndWork;
end;

procedure TApplicationScrobbler.PlaybackStarted;
begin
    GetSystemTime(StartTimeUTC);
    StartTimeLocal := Now;
    PlayAmount := 0;

    if DoScrobble then
    begin
        if not fWorking then
            // aktuelles File Scrobbeln
            ScrobbleCurrentFile
    end;
end;

procedure TApplicationScrobbler.PlaybackPaused;
begin
    PlayAmount := PlayAmount + SecondsBetween(Now, StartTimeLocal);
end;

procedure TApplicationScrobbler.PlaybackResumed;
begin
    StartTimeLocal := Now;
end;

procedure TApplicationScrobbler.AllowHandShakingAgain;
begin
    self.EarliestNextHandshake := Now;
end;


// -------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------
// Klasse TScrobbler
// -------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------


constructor TScrobbler.Create(aHandle: HWnd);
begin
    inherited Create;
    fWindowHandle := aHandle;
    fThread := 0;
    p  := '1.2.1'; // Protokoll-Version

    fIDHttp := TIdHttp.Create;
    fIDHttp.ConnectTimeout:= 20000;
    fIDHttp.ReadTimeout:= 20000;

    fIDHttp.Request.UserAgent := 'Mozilla/3.0';
    fIDHttp.HTTPOptions :=  [];  // Hinweis: hoForceEncodeParams maskiert keine '&',
                                 // was zu Problemen bei der einen oder anderen Band fhrt. ;-)

    EarliestNextHandshake := Now;
    ParamList := TStringList.Create;
end;

destructor TScrobbler.Destroy;
begin
    fIDHttp.Free;
    ParamList.Free;
    inherited destroy;
end;

function TScrobbler.GetUTCTimeStamp: AnsiString;
var SystemTime: TSystemTime;
    currentTime: TDateTime;
    diff: LongInt;
begin
    GetSystemTime(SystemTime);
    currentTime :=
        EncodeDateTime(SystemTime.wYear, SystemTime.wMonth, SystemTime.wDay, SystemTime.wHour, SystemTime.wMinute, SystemTime.wSecond, SystemTime.wMilliSeconds);
    diff := Round((currentTime - UnixStartDate) * 86400); // 86400: Sekunden pro Tag
    result := IntToStr(Diff);
end;


procedure TScrobbler.GetToken;
var Dummy: Cardinal;
begin
    fThread := BeginThread(Nil, 0, @StartGetToken, Self, 0, Dummy)
end;
procedure StartGetToken(Scrobbler: TScrobbler);
begin
    Scrobbler.fGetToken;
end;
procedure TScrobbler.fGetToken;
var Sig: AnsiString;
    Response, MessageText: AnsiString;
begin
  try
      Sig := 'api_key' + ApiKey
              + 'method' + 'auth.gettoken'
              + Secret;
      try
          Response := fIDHTTP.Get( BaseApiURL + '?method=auth.gettoken'
                            + '&' + 'api_key=' + ApiKey
                            + '&' + 'api_sig=' + Lowercase(MD5DigestToStr(MD5String(sig))));
          SendMessage(fWindowHandle, WM_Scrobbler, SC_GetToken, lParam(PAnsiChar(Response)));
      except
            on E: EIdHTTPProtocolException do
            begin
                MessageText := Scrobble_ProtocolError + #13#10 + 'Server message:'#13#10 + E.Message + #13#10 + Response;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetTokenException, lParam(PAnsiChar(MessageText)));
                //(z.B. 404)
            end;

            on E: EIdSocketError do
            begin
                MessageText := Scrobble_ConnectError + #13#10 + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetTokenException, lParam(PAnsiChar(MessageText)));
            end;

            on E: EIdexception do
            begin
                MessageText := Scrobble_UnkownError + '(' + E.ClassName + ') ' + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetTokenException, lParam(PAnsiChar(MessageText)));
            end;
        end;
     finally
        self.free;
     end;
end;


procedure TScrobbler.GetSession;
var Dummy: Cardinal;
begin
    fThread := BeginThread(Nil, 0, @StartGetSession, Self, 0, Dummy)
end;
procedure StartGetSession(Scrobbler: TScrobbler);
begin
    Scrobbler.fGetSession;
end;
procedure TScrobbler.fGetSession;
var Sig: AnsiString;
    Response, MessageText: AnsiString;
begin
    try
        Sig := 'api_key' + ApiKey
              + 'method' + 'auth.getsession'
              + 'token' + Token
              + Secret;
        try
            Response := fIDHTTP.Get( BaseApiURL + '?method=auth.getsession'
                            + '&' + 'api_key=' + ApiKey
                            + '&' + 'token=' + Token
                            + '&' + 'api_sig=' + Lowercase(MD5DigestToStr(MD5String(sig))));
            SendMessage(fWindowHandle, WM_Scrobbler, SC_GetSession, lParam(PAnsiChar(Response)));
        except
            on E: EIdHTTPProtocolException do
            begin
                MessageText := Scrobble_ProtocolError + #13#10 + 'Server message:'#13#10 + E.Message + #13#10 + Response;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetSessionException, lParam(PAnsiChar(MessageText)));
                //(z.B. 404)
            end;

            on E: EIdSocketError do
            begin
                MessageText := Scrobble_ConnectError + #13#10 + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetSessionException, lParam(PAnsiChar(MessageText)));
            end;

            on E: EIdexception do
            begin
                MessageText := Scrobble_UnkownError + '(' + E.ClassName + ') ' + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_GetSessionException, lParam(PAnsiChar(MessageText)));
            end;
        end;
    finally
        self.free;
    end;
end;

function TScrobbler.ParseHandShakeResult(aResponse: Ansistring): TScrobbleStatus;
var aList: TStringList;
    stat: AnsiString;
begin
    aList := TStringlist.Create;
    try
        aList.CommaText := aResponse;
        if (aList.Count >= 4) and (Uppercase(Trim(aList[0])) = 'OK') then
        begin
            SessionID := aList[1];
            NowPlayingURL := aList[2];
            SubmissionURL := aList[3];
            Result := hs_OK;
        end else
        begin
            if aList.Count = 0 then
                Result := hs_UnknownFailure
            else
            begin
                stat := Uppercase(Trim(aList[0]));
                if stat = 'BANNED' then
                  Result := hs_BANNED
                else
                  if stat = 'BADAUTH' then
                    Result := hs_BADAUTH
                  else
                    if stat = 'BADTIME' then
                      result := hs_BADTIME
                    else
                      if Pos('FAILED', Stat) > 0 then
                        result := hs_FAILED
                      else
                        result := hs_UnknownFailure
            end;
        end;
    finally
        aList.Free;
    end;
end;

function TScrobbler.fPerformHandShake: TScrobbleStatus;
var authToken: AnsiString;
    Response, MessageText: AnsiString;
    t: AnsiString;
begin
    if Now < EarliestNextHandshake then
    begin
        result := hs_HANDSHAKECANCELLED;
        SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeCancelled, lParam(MinutesBetween(Now, EarliestNextHandshake)+1 ));
        exit;
    end;

    if (Username = '') or (SessionKey = '') then
    begin
        // ABBRECHEN! User hat sich noch nicht angemeldet
        result := hs_UNAUTHORIZED;
        SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeError, lParam(Result));
        //
        //  Das ist ein sehr harter Fehler - Der User muss sich erst anmelden Alles weitere ist komplett sinnfrei.
        //
    end
    else
    begin
        t := GetUTCTimeStamp;
        authToken := Lowercase(MD5DigestToStr(MD5String(Secret + t)));
        try
            Response := fIDHTTP.Get( BaseScrobbleURL + '?hs=true'
                            + '&' + 'p=' + p
                            + '&' + 'c=' + c
                            + '&' + 'v=' + v
                            + '&' + 'u=' + Username
                            + '&' + 't=' + t
                            + '&' + 'a=' + authToken
                            + '&' + 'api_key=' + ApiKey
                            + '&' + 'sk=' + SessionKey);
            result := ParseHandShakeResult(Response);

            case result of
                // Wenn alles Ok war, sind damit SessionID, NowPlayingURL und SubmissionURL gesetzt
                hs_OK: SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeComplete, lParam(PAnsiChar(Response)));
                else
                begin
                    // Wenn nicht, dann ist der Handshake fehlgeschlagen.
                    SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeError, lParam(Result));
                    if result = hs_failed then
                        // Some Failure => Text senden
                        SendMessage(fWindowHandle, WM_Scrobbler, SC_Message, lParam(Response));
                end;
            end;

        except
            on E: EIdHTTPProtocolException do
            begin
                result := hs_Exception;
                MessageText := Scrobble_ProtocolError + #13#10 + 'Server message:' + #13#10 + E.Message + #13#10 + Response;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeException, lParam(PAnsiChar(MessageText)));
                //(z.B. 404)
            end;

            on E: EIdSocketError do
            begin
                result := hs_Exception;
                MessageText := Scrobble_ConnectError + #13#10 + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeException, lParam(PAnsiChar(MessageText)));
            end;

            on E: EIdexception do
            begin
                result := hs_Exception;
                MessageText := Scrobble_UnkownError + '(' + E.ClassName + ') ' + E.Message;
                SendMessage(fWindowHandle, WM_Scrobbler, SC_HandShakeException, lParam(PAnsiChar(MessageText)));
            end;
        end;
    end;
end;


function TScrobbler.ParseNowPlayingResult(aResponse: Ansistring): TScrobbleStatus;
var stat: AnsiString;
begin
    stat := Uppercase(Trim(aResponse));
    if stat = 'OK' then
      result := hs_OK
    else
      if stat = 'BADSESSION' then
        result := hs_BADSESSION
      else
        result := hs_UnknownFailure;
end;

function TScrobbler.fPerformNowPlayingNotification: TScrobbleStatus;
var Response, MessageText: AnsiString;
begin
    try
        ParamList[0] := 's=' + SessionID;
        Response := fIDHTTP.Post(NowPlayingURL, ParamList);
        result := ParseNowPlayingResult(Response);
    except

        on E: EIdHTTPProtocolException do
        begin
            result := hs_Exception;
            MessageText := Scrobble_ProtocolError + #13#10 + 'Server message:'#13#10 + E.Message + #13#10 + Response;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingException, lParam(PAnsiChar(MessageText)));
            //(z.B. 404)
        end;

        on E: EIdSocketError do
        begin
            result := hs_Exception;
            MessageText := Scrobble_ConnectError + #13#10 + E.Message;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingException, lParam(PAnsiChar(MessageText)));
        end;

        on E: EIdexception do
        begin
            result := hs_Exception;
            MessageText := Scrobble_UnkownError + '(' + E.ClassName + ') ' + E.Message;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingException, lParam(PAnsiChar(MessageText)));
        end;
    end;
end;


procedure TScrobbler.ScrobbleNowPlaying;
var Dummy: Cardinal;
begin
    fThread := BeginThread(Nil, 0, @StartScrobbleNowPlaying, Self, 0, Dummy)
end;
procedure StartScrobbleNowPlaying(Scrobbler: TScrobbler);
begin
    Scrobbler.fScrobbleNowPlaying;
end;
procedure TScrobbler.fScrobbleNowPlaying;
var tmpStat: TScrobbleStatus;
begin
    try
        if SessionID = '' then
            tmpStat := fPerformHandShake
        else
            tmpStat := hs_OK;

        if tmpStat = hs_Ok then
        begin
            tmpStat := fPerformNowPlayingNotification;
            if tmpStat = hs_Ok then
                // Scrobbeln erfolgreich
                SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingComplete, LParam(PAnsiChar(SuccessMessage)))
            else
            begin
                // Scrobbeln war nicht erfolgreich. d.h.: (alte) Session-ID ist ungltig.
                // also: Neuer Handshake und zweiter Versuch.
                if tmpStat = hs_BADSESSION then
                    tmpStat := fPerformHandShake
                else
                    tmpStat := hs_OK; // einfach neu probieren
                if tmpStat = hs_Ok then
                begin
                    // Handshake war ok, nochmal Scrobbeln
                    tmpStat := fPerformNowPlayingNotification;
                    if tmpStat = hs_Ok then
                        // Scrobbeln erfolgreich
                        SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingComplete, LParam(PAnsiChar(SuccessMessage)))
                    else
                        SendMessage(fWindowHandle, WM_Scrobbler, SC_NowPlayingError, lParam(tmpStat));
                end;
            end;
        end;
    finally
        self.free;
    end;
end;


function TScrobbler.ParseSubmissionResult(aResponse: Ansistring): TScrobbleStatus;
var stat: AnsiString;
begin
    stat := Uppercase(Trim(aResponse));
    if stat = 'OK' then
      result := hs_OK
    else
      if stat = 'BADSESSION' then
        result := hs_BADSESSION
      else
        if Pos('FAILED', Stat) > 0 then
          result := hs_FAILED
        else
          result := hs_UnknownFailure;
end;

function TScrobbler.fPerformSubmissionNotification: TScrobbleStatus;
var Response, MessageText: AnsiString;
begin
    try
        ParamList[0] := 's=' + SessionID;
        Response := fIDHTTP.Post(SubmissionURL, ParamList);
        result := ParseSubmissionResult(Response);

        if result = hs_failed then
        // Some Failure => Text senden
            SendMessage(fWindowHandle, WM_Scrobbler, SC_Message, lParam(Response));

    except
        on E: EIdHTTPProtocolException do
        begin
            result := hs_Exception;
            MessageText := Scrobble_ProtocolError + #13#10 + 'Server message: '#13#10 + E.Message + #13#10 + Response;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionException, lParam(PAnsiChar(MessageText)));
            //(z.B. 404)
        end;

        on E: EIdSocketError do
        begin
            result := hs_Exception;
            MessageText := Scrobble_ConnectError + #13#10 + E.Message;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionException, lParam(PAnsiChar(MessageText)));
        end;

        on E: EIdexception do
        begin
            result := hs_Exception;
            MessageText := Scrobble_UnkownError + '(' + E.ClassName + ') ' + E.Message;
            SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionException, lParam(PAnsiChar(MessageText)));
        end;
    end;
end;

procedure TScrobbler.ScrobbleSubmit;
var Dummy: Cardinal;
begin
    fThread := BeginThread(Nil, 0, @StartScrobbleSubmit, Self, 0, Dummy)
end;
procedure StartScrobbleSubmit(Scrobbler: TScrobbler);
begin
    Scrobbler.fScrobbleSubmit;
end;
procedure TScrobbler.fScrobbleSubmit;
var tmpStat: TScrobbleStatus;
begin
    try
        if SessionID = '' then
            tmpStat := fPerformHandShake
        else
            tmpStat := hs_OK;

        if tmpStat = hs_Ok then
        begin
            tmpStat := fPerformSubmissionNotification;
            if tmpStat = hs_Ok then
                // Scrobbeln erfolgreich
                SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionComplete, LParam(PAnsiChar(SuccessMessage)))
            else
            begin
                // Scrobbeln war nicht erfolgreich. d.h.: (alte) Session-ID ist ungltig.
                // also: Neuer Handshake und zweiter Versuch.
                if tmpStat = hs_BADSESSION then
                    tmpStat := fPerformHandShake
                else
                    tmpStat := hs_OK; // einfach neu probieren

                if tmpStat = hs_Ok then
                begin
                    // Handshake war ok, nochmal Scrobbeln
                    tmpStat := fPerformSubmissionNotification;
                    if tmpStat = hs_Ok then
                        // Scrobbeln erfolgreich
                        SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionComplete, LParam(PAnsiChar(SuccessMessage)))
                    else
                        SendMessage(fWindowHandle, WM_Scrobbler, SC_SubmissionError, lParam(tmpStat));
                end;
            end;
        end;
    finally
        self.free;
    end;
end;

end.


